import './bootstrap';
import './bbcode';
import './interactive-map';

$(document).ready(function() {
    initNavigation();
    initCounters();
    
    // Initialize notifications if user is authenticated
    if (window.notificationRoutes) {
        initNotifications();
    }
});

function initNavigation() {
    // Mobile navigation toggle
    $('.mobile-nav-toggle').on('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        const $mainNav = $('#mainNav');
        const $overlay = $('#mobileMenuOverlay');
        const $icon = $(this).find('i');
        const isOpen = $mainNav.hasClass('show');
        
        if (isOpen) {
            // Close menu
            $mainNav.removeClass('show');
            $overlay.removeClass('show');
            $icon.removeClass('fa-times').addClass('fa-bars');
            $(this).removeClass('active').attr('aria-expanded', 'false');
            $('body').css('overflow', '');
        } else {
            // Open menu
            $mainNav.addClass('show');
            $overlay.addClass('show');
            $icon.removeClass('fa-bars').addClass('fa-times');
            $(this).addClass('active').attr('aria-expanded', 'true');
            $('body').css('overflow', 'hidden');
        }
    });

    // Close mobile menu when clicking overlay
    $('#mobileMenuOverlay').on('click', function() {
        closeMobileMenu();
    });

    // Mobile dropdown toggles
    $('.nav-dropdown > a').on('click', function(e) {
        // Only handle clicks on mobile
        if (window.innerWidth <= 768) {
            e.preventDefault();
            const $parent = $(this).parent();
            const $menu = $parent.find('.nav-dropdown-menu');
            
            // Close other dropdowns
            $('.nav-dropdown-menu.show').not($menu).removeClass('show');
            $('.nav-dropdown').not($parent).removeClass('open');
            
            // Toggle this dropdown
            $menu.toggleClass('show');
            $parent.toggleClass('open');
        }
    });

    // User dropdown functionality
    let userDropdownTimeout;
    
    // Desktop hover behavior
    $('.user-dropdown').on('mouseenter', function() {
        if (window.innerWidth > 768) {
            clearTimeout(userDropdownTimeout);
            $(this).addClass('hover');
        }
    });
    
    $('.user-dropdown').on('mouseleave', function() {
        if (window.innerWidth > 768) {
            const $dropdown = $(this);
            userDropdownTimeout = setTimeout(function() {
                $dropdown.removeClass('hover');
            }, 200);
        }
    });

    // Click handler for user dropdown
    $('.user-dropdown .user-info').on('click', function(e) {
        e.stopPropagation();
        const $dropdown = $(this).closest('.user-dropdown');
        const $menu = $dropdown.find('.dropdown-menu');
        
        if (window.innerWidth <= 768) {
            // Mobile: toggle class on parent
            $dropdown.toggleClass('mobile-dropdown-open');
        } else {
            // Desktop: toggle show class on menu
            $menu.toggleClass('show');
        }
    });

    // Close dropdowns and mobile menu when clicking outside
    $(document).on('click', function(e) {
        // Close user dropdown
        if (!$(e.target).closest('.user-dropdown').length) {
            $('.user-dropdown').removeClass('mobile-dropdown-open hover');
            $('.user-dropdown .dropdown-menu').removeClass('show');
        }
        
        // Close mobile navigation
        if (!$(e.target).closest('.main-nav').length && !$(e.target).closest('.mobile-nav-toggle').length) {
            closeMobileMenu();
        }
        
        // Close mobile nav dropdowns
        if (!$(e.target).closest('.nav-dropdown').length) {
            $('.nav-dropdown').removeClass('open');
            $('.nav-dropdown-menu').removeClass('show');
        }
    });

    // Handle window resize
    $(window).on('resize', function() {
        if (window.innerWidth > 768) {
            // Reset mobile states on desktop
            closeMobileMenu();
            $('.nav-dropdown').removeClass('open');
            $('.nav-dropdown-menu').removeClass('show');
            $('.user-dropdown').removeClass('mobile-dropdown-open');
        } else {
            // Reset desktop states on mobile
            $('.user-dropdown').removeClass('hover');
            $('.user-dropdown .dropdown-menu').removeClass('show');
        }
    });

    // Prevent dropdown menu clicks from closing the dropdown
    $('.dropdown-menu, .nav-dropdown-menu').on('click', function(e) {
        e.stopPropagation();
    });

    // Handle escape key to close mobile menu and dropdowns
    $(document).on('keydown', function(e) {
        if (e.key === 'Escape') {
            closeMobileMenu();
            $('.user-dropdown').removeClass('mobile-dropdown-open hover');
            $('.user-dropdown .dropdown-menu').removeClass('show');
            $('.nav-dropdown').removeClass('open');
            $('.nav-dropdown-menu').removeClass('show');
        }
    });

    // Navigation dropdown hover for desktop
    $('.nav-dropdown').on('mouseenter', function() {
        if (window.innerWidth > 768) {
            $(this).addClass('hover');
        }
    });

    $('.nav-dropdown').on('mouseleave', function() {
        if (window.innerWidth > 768) {
            $(this).removeClass('hover');
        }
    });

    // Close mobile menu when clicking on a direct link (not dropdowns)
    $('.main-nav a:not(.nav-dropdown > a)').on('click', function() {
        if (window.innerWidth <= 768) {
            closeMobileMenu();
        }
    });
}

// Helper function to close mobile menu
function closeMobileMenu() {
    const $mainNav = $('#mainNav');
    const $overlay = $('#mobileMenuOverlay');
    const $toggle = $('.mobile-nav-toggle');
    const $icon = $toggle.find('i');
    
    $mainNav.removeClass('show');
    $overlay.removeClass('show');
    $icon.removeClass('fa-times').addClass('fa-bars');
    $toggle.removeClass('active').attr('aria-expanded', 'false');
    $('body').css('overflow', '');
}

// Counter animation for stats
function initCounters() {
    $('.stat-value').each(function() {
        const $this = $(this);
        const countTo = parseInt($this.text(), 10);
        
        // Only animate if it's a valid number
        if (!isNaN(countTo)) {
            $({ countNum: 0 }).animate({
                countNum: countTo
            }, {
                duration: 2000,
                easing: 'swing',
                step: function() {
                    $this.text(Math.floor(this.countNum));
                },
                complete: function() {
                    $this.text(this.countNum);
                }
            });
        }
    });
}

// ============================================
// NOTIFICATION FUNCTIONALITY
// ============================================
function initNotifications() {
    let notificationsLoaded = false;
    let dropdownOpen = false;

    // Toggle notification dropdown
    $('.header-notification-bell').on('click', function(e) {
        e.stopPropagation();
        const $dropdown = $('#header-notification-dropdown');
        
        dropdownOpen = !dropdownOpen;
        
        if (dropdownOpen) {
            $dropdown.addClass('is-visible');
            if (!notificationsLoaded) {
                loadNotifications();
            }
        } else {
            $dropdown.removeClass('is-visible');
        }
    });

    // Close notification dropdown when clicking outside
    $(document).on('click', function(e) {
        if (!$(e.target).closest('#header-notification-container').length) {
            $('#header-notification-dropdown').removeClass('is-visible');
            dropdownOpen = false;
        }
    });

    // Mark all as read
    $('.header-mark-all-read').on('click', function(e) {
        e.preventDefault();
        markAllNotificationsAsRead();
    });

    // Load notifications
    function loadNotifications() {
        const $container = $('#header-notification-list');
        
        $.ajax({
            url: window.notificationRoutes.recent,
            method: 'GET',
            success: function(data) {
                if (!Array.isArray(data.notifications)) {
                    throw new Error('Invalid data format');
                }
                
                if (data.notifications.length === 0) {
                    $container.html(`
                        <div class="header-notification-empty">
                            <div class="header-empty-icon">🔔</div>
                            <div class="header-empty-title">No notifications</div>
                            <div class="header-empty-description">You're all caught up!</div>
                        </div>
                    `);
                } else {
                    $container.empty();
                    
                    data.notifications.forEach(function(notification) {
                        const $item = createNotificationItem(notification);
                        $container.append($item);
                    });
                }
                
                updateNotificationBadges(data.unread_count || 0);
                notificationsLoaded = true;
            },
            error: function(error) {
                console.error('Error loading notifications:', error);
                $container.html(`
                    <div class="header-notification-empty">
                        <div class="header-empty-icon">⚠️</div>
                        <div class="header-empty-title">Error loading notifications</div>
                        <div class="header-empty-description">Please try again later</div>
                    </div>
                `);
            }
        });
    }

    // Create notification item (XSS-safe)
    function createNotificationItem(notification) {
        const $item = $('<a>')
            .addClass('header-notification-item')
            .attr('href', sanitizeUrl(notification.data?.url || '#'));
        
        if (!notification.read_at) {
            $item.addClass('is-unread');
        }
        
        $item.on('click', function(e) {
            markNotificationAsRead(notification.id, e);
        });
        
        const $content = $('<div>').addClass('header-notification-content');
        
        // Avatar
        const avatar = notification.data?.reactor_avatar || notification.data?.replier_avatar;
        if (avatar) {
            const $avatar = $('<img>')
                .addClass('header-notification-avatar')
                .attr('src', sanitizeUrl(avatar))
                .attr('alt', 'Avatar');
            $content.append($avatar);
        } else {
            const $defaultAvatar = $('<div>')
                .addClass('header-notification-avatar default-avatar')
                .html(`<svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"></path>
                    <circle cx="12" cy="7" r="4"></circle>
                </svg>`);
            $content.append($defaultAvatar);
        }
        
        // Body
        const $body = $('<div>').addClass('header-notification-body');
        
        // Message
        const $message = $('<div>')
            .addClass('header-notification-message')
            .text(notification.data?.message || '');
        
        if (notification.data?.reaction_emoji) {
            const $emoji = $('<span>')
                .addClass('header-reaction-emoji')
                .text(notification.data.reaction_emoji);
            $message.append($emoji);
        }
        $body.append($message);
        
        // Preview
        if (notification.data?.post_content_preview) {
            const $preview = $('<div>')
                .addClass('header-notification-preview')
                .text(`"${notification.data.post_content_preview}"`);
            $body.append($preview);
        }
        
        // Time
        const $time = $('<div>')
            .addClass('header-notification-time')
            .text(formatTimeAgo(notification.created_at));
        $body.append($time);
        
        $content.append($body);
        $item.append($content);
        
        return $item;
    }

    // Mark notification as read
    function markNotificationAsRead(notificationId, event) {
        if (event) {
            event.preventDefault();
        }
        
        const sanitizedId = String(notificationId).replace(/[^a-zA-Z0-9-]/g, '');
        if (!sanitizedId) return;
        
        const url = window.notificationRoutes.markRead.replace(':id', sanitizedId);
        
        $.ajax({
            url: url,
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function() {
                notificationsLoaded = false;
                loadNotifications();
                
                if (event) {
                    const href = $(event.target).closest('.header-notification-item').attr('href');
                    if (href && href !== '#') {
                        window.location.href = sanitizeUrl(href);
                    }
                }
            },
            error: function(error) {
                console.error('Error marking notification as read:', error);
            }
        });
    }

    // Mark all notifications as read
    function markAllNotificationsAsRead() {
        $.ajax({
            url: window.notificationRoutes.markAllRead,
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
            },
            success: function() {
                notificationsLoaded = false;
                loadNotifications();
            },
            error: function(error) {
                console.error('Error marking all notifications as read:', error);
            }
        });
    }

    // Update notification badges
    function updateNotificationBadges(count) {
        const safeCount = Math.max(0, parseInt(count) || 0);
        
        const $bellBadge = $('#header-notification-badge');
        if ($bellBadge.length) {
            $bellBadge.text(safeCount);
            $bellBadge.toggleClass('is-hidden', safeCount === 0);
        }
        
        const $dropdownBadge = $('#dropdown-notification-badge');
        if ($dropdownBadge.length) {
            $dropdownBadge.text(safeCount);
            $dropdownBadge.css('display', safeCount > 0 ? 'inline-flex' : 'none');
        }
    }

    // Load initial unread count
    $.ajax({
        url: window.notificationRoutes.unreadCount,
        method: 'GET',
        success: function(data) {
            updateNotificationBadges(data.count || 0);
        },
        error: function(error) {
            console.error('Error loading initial notification count:', error);
        }
    });

    // Auto-refresh unread count every 30 seconds
    setInterval(function() {
        $.ajax({
            url: window.notificationRoutes.unreadCount,
            method: 'GET',
            success: function(data) {
                updateNotificationBadges(data.count || 0);
                if (dropdownOpen) {
                    notificationsLoaded = false;
                }
            },
            error: function(error) {
                console.error('Error updating notification count:', error);
            }
        });
    }, 30000);
}

// ============================================
// UTILITY FUNCTIONS
// ============================================

// XSS-Safe URL sanitization
function sanitizeUrl(url) {
    if (!url) return '#';
    const trimmed = String(url).trim();
    if (trimmed.startsWith('/') || trimmed.startsWith('http://') || trimmed.startsWith('https://')) {
        return trimmed;
    }
    return '#';
}

// Format time ago
function formatTimeAgo(dateString) {
    try {
        const date = new Date(dateString);
        if (isNaN(date.getTime())) return 'Recently';
        
        const now = new Date();
        const seconds = Math.floor((now - date) / 1000);
        
        const intervals = {
            year: 31536000,
            month: 2592000,
            week: 604800,
            day: 86400,
            hour: 3600,
            minute: 60
        };
        
        for (const [unit, secondsInUnit] of Object.entries(intervals)) {
            const interval = Math.floor(seconds / secondsInUnit);
            if (interval >= 1) {
                return `${interval} ${unit}${interval !== 1 ? 's' : ''} ago`;
            }
        }
        
        return 'Just now';
    } catch (e) {
        return 'Recently';
    }
}